<?php
// Include the database connection file
require_once 'db_connect.php';

// ----------------------
// 1. DATABASE CONNECTION
// ----------------------
// Create a new PDO instance to connect to MySQL
// host: 127.0.0.1, database: cardan, charset: utf8mb4, user: root, pass: ''
try {
    $pdo = new PDO(
        'mysql:host=127.0.0.1;dbname=cardan;charset=utf8mb4',
        'root',
        ''
    );
    // Throw exceptions on database errors
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
} catch (PDOException $e) {
    // If connection fails, stop script and show error
    exit('Connection error: ' . $e->getMessage());
}

// ----------------------
// 2. FETCH PAINTS & DETAILS
// ----------------------
// Retrieve first 10 paints along with their main color name
$sql = "
    SELECT p.id, p.name, p.price, p.grams,
           p.shine_level, p.base_type, p.photo,
           p.main_color_id, mc.name AS main_color_name
      FROM paint_types p
      JOIN main_colors mc ON mc.id = p.main_color_id
     WHERE p.id BETWEEN 1 AND 10
     ORDER BY p.id
";
$paints = $pdo->query($sql)->fetchAll(PDO::FETCH_ASSOC);

// ----------------------
// 3. PRELOAD RAL COLORS & TIPS
// ----------------------
// Load all RAL colors from database
$allRal = $pdo->query("SELECT * FROM ral_colors")->fetchAll(PDO::FETCH_ASSOC);
$ralBy = [];
// Group RALs by their main_color_id for quick lookup
foreach ($allRal as $r) {
    $ralBy[$r['main_color_id']][] = $r;
}

// Load all tips from database
$allTips = $pdo->query("SELECT main_color_id, tip_text FROM tips")->fetchAll(PDO::FETCH_ASSOC);
$tipsBy = [];
// Group tips by main_color_id as well
foreach ($allTips as $t) {
    $tipsBy[$t['main_color_id']][] = $t['tip_text'];
}

// ----------------------
// 4. QUIZ OPTIONS GENERATOR
// ----------------------
/**
 * Given RALs for one main color and all groups, pick 1 correct + 3 incorrect
 * @param array $group RALs for the current paint's main color
 * @param array $all   All RAL groups keyed by main_color_id
 * @param int   $id    Current main_color_id
 * @return array ['opts' => [ { num, hex } x4 ], 'correct' => correctNumber]
 */
function buildOptions(array $group, array $all, int $id): array {
    // Find the correct RAL (marked by is_correct flag)
    $correctList = array_filter($group, fn($r) => !empty($r['is_correct']));
    // If none marked, pick one randomly
    $correct = $correctList ? reset($correctList) : $group[array_rand($group)];

    // Collect 3 wrong answers from other main-color groups
    $incorrect = [];
    $others = array_diff_key($all, [$id => 1]);
    while (count($incorrect) < 3) {
        // Pick a random group, then a random RAL
        $g = $others[array_rand($others)];
        $pick = $g[array_rand($g)];
        // Ensure we don't duplicate the correct one
        if ($pick['ral_number'] !== $correct['ral_number']) {
            $incorrect[$pick['ral_number']] = $pick;
        }
    }

    // Merge correct + incorrect and shuffle
    $options = array_merge([$correct], array_values($incorrect));
    shuffle($options);

    // Return only number and hex for display
    return [
        'opts'    => array_map(
            fn($r) => ['num' => $r['ral_number'], 'hex' => $r['hex'] ?? '#CCC'],
            $options
        ),
        'correct' => $correct['ral_number'],
    ];
}

// ----------------------
// 5. BUILD QUIZ SLIDES
// ----------------------
$slides = [];
foreach ($paints as $p) {
    // Generate swatch options for this paint
    $opt = buildOptions(
        $ralBy[$p['main_color_id']] ?? [],
        $ralBy,
        $p['main_color_id']
    );
    // Combine paint info + quiz options into one slide
    $slides[] = array_merge(
        [
            'id'            => $p['id'],
            'paintName'     => $p['name'],
            'paintPhoto'    => $p['photo'],
            'price'         => $p['price'],
            'grams'         => $p['grams'],
            'shine'         => $p['shine_level'],
            'base'          => $p['base_type'],
            'mainColorName' => $p['main_color_name'],
            'tips'          => $tipsBy[$p['main_color_id']] ?? [],
        ],
        $opt
    );
}
// Randomize slide order
shuffle($slides);
?>

<!DOCTYPE html>
<html lang="en" class="tritanopia">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1">
    <title>Paint & RAL Store</title>
    <link rel="stylesheet" href="stylesheet.css">
    <script>
        // Make slides data available to JavaScript
        window.slides = <?php echo json_encode($slides); ?>;
    </script>
    <script src="JavaScript.js" defer></script>
    
</head>
    <header>
        
        <nav>
        <ul id="top-nav" class="nav">
            
                <h1 >Verfwereld</h1>
                <div class="cardan-text">
                <li><a href="#quit-simulation">Stop simulatie</a></li>
                <li><a href="#mijnCardan">Mijn Cardan</a></li>
                <li><a href="#cookies">Cookies</a></li>
            </div>

            </ul>
        <div id="big-nav" >
        <img src="#" alt="Logo Cardan">
        
            <ul id="bottom-nav" class="nav">
                <li><a href="#our-services">Onze diensten</a></li>
                <li><a href="#branches">Branches</a></li>
                <li><a href="#Kennisbank">Kennisbank</a></li>
                <li><a href="#about-us">Over ons</a></li>
                <li><a href="#contact">Contact</a></li>
        </ul>
        <ul id="language-nav" class="nav">
            <li><a href="#nl">NL</a></li>
            <p>|</p>
            <li><a href="#en">EN</a></li>
        </ul>
        </div>
    </nav>
        <div class="Head-Banner">
            <div class="Banner-card-ad-1">advertisement</div>
            <div class="Banner-card-ad-2">try other filters</div>
            <div class="Banner-card-ad-3">try other filters</div>
            <div class="Banner-card-ad-4">back to cardan</div>
        </div>
    </header>
    
    <body>
        
    </body>
    
</html>

